'use client';

import cn from 'classnames';
import { useEffect, useState } from 'react';
import ModelsList from './ui/ModelsList';
import './index.scss';
import ArticlesList from './ui/ArticlesList';
import { redirect, useSearchParams } from 'next/navigation';
import { formatTime, getPublicStrapiContent } from '@/app/lib/utils';
import { Article, Model } from './model/interface';
import { formatDate } from 'react-datepicker/dist/date_utils';

const Results = ({ lang, translates }: any) => {
  const searchParams = useSearchParams();
  const [searchParam, setSearchParam] = useState('');

  const [{ articles, models }, setSearchData] = useState<{
    articles: Article[];
    models: Model[];
  }>({
    articles: [],
    models: [],
  });

  const getData = async () => {
    setShowLoading(true);
    try {
      const [articlesData, modelsData] = await Promise.all([
        getPublicStrapiContent({
          url: `/api/articles?locale=${lang}&publicationState=live&filters[$or][0][previewTitle][$containsi]=${searchParam}&filters[$or][1][previewDescription][$containsi]=${searchParam}&populate=deep`,
        }),
        getPublicStrapiContent({
          url: `/api/product/search/${searchParam}?locale=${lang}&publicationState=live&populate[0]=CarouselImage`,
        }),
      ]);

      const result: {
        articles: Article[];
        models: Model[];
      } = { articles: [], models: [] };

      if (articlesData.data) {
        result.articles = articlesData.data.map(({ attributes }: any) => ({
          title: attributes.previewTitle,
          description: attributes.previewDescription,
          img: attributes.previewImg?.data?.attributes,
          url: `/${lang}/news/${attributes.slug}`,
          data: formatTime(attributes.updatedAt),
        }));
      }

      if (modelsData.data) {
        result.models = modelsData.data.map(({ attributes }: any) => ({
          name: attributes.name,
          slug: attributes.slug,
          img: attributes.CarouselImage.data.attributes,
        }));
      }

      setSearchData(result);
      setShowLoading(false);
    } catch (e) {
      console.log(e);
    }
  };

  useEffect(() => {
    const search = searchParams?.get('q');

    if (!search) {
      redirect(`/${lang}`);
    }
    setSearchParam(search);
  }, [searchParams]);

  useEffect(() => {
    setShowLoading(true);
    searchParam && getData();
  }, [searchParam]);

  const [showLoading, setShowLoading] = useState(true);

  const [activeTab, setActiveTab] = useState<'articles' | 'models'>('models');
  const tabs = [
    { title: translates?.articlesTab || 'Articles', name: 'articles' },
    { title: translates?.productsTab || 'Products', name: 'models' },
  ];

  const resultsContent = () => {
    switch (activeTab) {
      case 'articles':
        return <ArticlesList articles={articles} translates={translates} />;
      case 'models':
        return <ModelsList lang={lang} models={models} />;
    }
  };

  const handleTabClick = (type: 'articles' | 'models') => {
    if (activeTab !== type) {
      setActiveTab(type);
    }
  };

  return (
    <section className="results">
      <div className="container">
        <div className="row">
          <div className="col-12 col-lg-4">
            {!showLoading ? (
              <span
                className="search-results__title typo-head-2"
                dangerouslySetInnerHTML={{
                  __html: (
                    translates?.searchTitle ||
                    '{{amount}} search results for {{searchString}}'
                  )
                    .replace(
                      '{{amount}}',
                      (models?.length || 0) + (articles?.length || 0),
                    )
                    .replace('{{searchString}}', `<u>${searchParam}</u>`),
                }}
              ></span>
            ) : (
              <span className="search-results__title typo-head-2">
                Loading...
              </span>
            )}
          </div>
          <div className="col-12 col-lg-8">
            {!showLoading && (
              <div className="search-results">
                <div className="search-results__container">
                  <div className="search-results__tabs-container">
                    <div className="search-results__tabs">
                      {tabs.map(({ name, title }) => (
                        <span
                          key={name}
                          className={cn('tab', {
                            tab_active: name === activeTab,
                          })}
                          onClick={() => handleTabClick(name as any)}
                        >
                          {/* @ts-ignore */}
                          {title} (
                          {(name === 'articles' ? articles : models)?.length ||
                            0}
                          )
                        </span>
                      ))}
                    </div>
                  </div>

                  <div className="search-results__content">
                    {resultsContent()}
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </section>
  );
};

export default Results;
