import AspectRatio from '@/shared/ui/AspectRatio';
import { StrapiImage, StrapiImageInterface } from '@/shared/ui/StrapiImage';
import cn from 'classnames';
import React, { PropsWithChildren } from 'react';
import './index.scss';

interface Props {
  desktopImageAspectRatio: string;
  mobileImageAspectRatio: string;
  sectionStyles: string;
  hideImageInMobile: boolean;
  hideImageInDesktop: boolean;
  fullWidth: boolean;
  mobileImage?: { data?: { attributes: StrapiImageInterface } };
  desktopImage?: { data?: { attributes: StrapiImageInterface } };
}

const getWidthAndHeight = (
  value: string,
  defaultValue: string = '1440/700',
): [number, number] => {
  let width = 0;
  let height = 0;

  try {
    let [w, h] = value.split('/');
    width = +w;
    height = +h;
  } catch (e) {
    const [w, h] = getWidthAndHeight(defaultValue);

    width = +w;
    height = +h;
  }

  if (!Number(width) || !Number(height)) {
    return getWidthAndHeight(defaultValue);
  }

  return [width, height];
};

const ImageSection = ({
  desktopImageAspectRatio,
  mobileImageAspectRatio,
  fullWidth,
  desktopImage,
  mobileImage,
  sectionStyles,
  hideImageInDesktop,
  hideImageInMobile,
}: Props) => {
  const [desktopWidth, desktopHeight] = getWidthAndHeight(
    desktopImageAspectRatio,
    '1440/445',
  );

  const [mobileWidth, mobileHeight] = getWidthAndHeight(
    mobileImageAspectRatio,
    '375/200',
  );

  const Content = ({ children }: PropsWithChildren) => {
    return !fullWidth ? (
      <div className="container">
        <div className="row">
          <div className="col-12">{children}</div>
        </div>
      </div>
    ) : (
      <>{children}</>
    );
  };

  return (
    <>
      {sectionStyles && <style>{` .image-section {${sectionStyles}} `}</style>}
      <section className="image-section">
        <div className="image-section__container">
          <Content>
            <AspectRatio
              width={desktopWidth}
              height={desktopHeight}
              mobileWidth={mobileWidth}
              mobileHeight={mobileHeight}
              className={cn({
                'd-lg-none-up': hideImageInDesktop,
                'd-lg-none-down': hideImageInMobile,
              })}
            >
              {desktopImage?.data?.attributes && (
                <StrapiImage
                  img={desktopImage?.data?.attributes}
                  className={cn({
                    'd-lg-none-down': !!mobileImage?.data?.attributes,
                  })}
                />
              )}

              {mobileImage?.data?.attributes && (
                <StrapiImage
                  img={mobileImage?.data?.attributes}
                  className="d-lg-none-up"
                />
              )}
            </AspectRatio>
          </Content>
        </div>
      </section>
    </>
  );
};

export default ImageSection;
