'use client';

import { useState, useRef, useEffect } from 'react';
import Link from 'next/link';
import cn from 'classnames';

import useMediaQuery from '@/app/lib/use-media-query';

import { Arrow, CheryLogo, Loupe } from '@icons';
import { Header as IHeader } from './model/interface';
import Language from './ui/Language/index';
import Search from './ui/Search/index';
import { Burger } from './ui';
import ModelsList from './ui/ModelsList';

import './index.scss';
import MobileLanguage from './ui/MobileLanguage';
import MobileCtaButtons from './ui/MobileCtaButtons';
import FindDealer from './ui/FindDealer';

const Header = ({
  lang,
  type = 'light',
  data,
  models,
  locales,
  hideHeaderOnScroll,
  translates,
  offerTypeLayout,
  eventTypeLayout,
}: IHeader) => {
  const headerRef = useRef<HTMLElement>(null);

  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [screenType, setScreenType] = useState<'models' | 'search' | ''>('');
  const [scrollPosition, setScrollPosition] = useState<number>(0);
  const isDesktop = useMediaQuery('(min-width: 1024px)');
  const [initModels, setInitModels] = useState(false);

  const handleScroll = () => {
    if (headerRef.current) {
      setScrollPosition(window.scrollY);
    }
  };

  const handleMenuClick = (state: boolean) => {
    setIsMenuOpen(state);
    setScreenType('');
    updateHeaderHeightVar();
  };

  const handleModelsClick = () => {
    setScreenType('models');
    setTimeout(() => {
      updateHeaderHeightVar();
    }, 0);
  };

  const handleBackButton = () => {
    setScreenType('');
    setTimeout(() => {
      updateHeaderHeightVar();
    }, 0);
  };

  const updateHeaderHeightVar = () => {
    if (headerRef.current) {
      document.body.style.setProperty(
        '--header-height',
        `${headerRef.current.offsetHeight}px`,
      );
    }
  };

  const handleResize = () => {
    updateHeaderHeightVar();
  };

  const handleSearchOpen = (isOpen: boolean) => {
    setScreenType(isOpen ? 'search' : '');
    setTimeout(() => {
      updateHeaderHeightVar();
    }, 0);
  };

  const onModelsEnter = () => {
    if (!isDesktop && screenType !== 'models') {
      return;
    }

    setScreenType('models');
  };

  const onModelsLeave = () => {
    if (!isDesktop || screenType !== 'models') {
      return;
    }
    setScreenType('');
  };

  useEffect(() => {
    ['html', 'body'].forEach((selector, index) => {
      const item = document.querySelector(selector) as HTMLElement;

      if (item) {
        if (isMenuOpen) {
          item.style.overflow = 'hidden';
          item.style.touchAction = 'none';
        } else {
          item.style.removeProperty('overflow');
          item.style.removeProperty('touch-action');
        }
      }
    });
  }, [isMenuOpen]);

  useEffect(() => {
    updateHeaderHeightVar();
    window.addEventListener('resize', handleResize);
    window.addEventListener('scroll', handleScroll);

    handleScroll();

    return () => {
      window.removeEventListener('resize', handleResize);
      window.removeEventListener('scroll', handleScroll);
    };
  }, []);

  useEffect(() => {
    if (screenType === 'models' && !initModels) {
      setInitModels(true);
    }
  }, [screenType]);

  const isHeaderLight =
    type === 'light' ||
    (type === 'transparent' &&
      ((isMenuOpen && screenType !== 'search') || screenType === 'models')) ||
    (!isDesktop && screenType !== '') ||
    (scrollPosition > 0 && !offerTypeLayout);

  const isHeaderTransparent =
    type === 'transparent' &&
    !(isMenuOpen || screenType === 'models') &&
    scrollPosition === 0;

  return (
    <header
      ref={headerRef}
      className={cn('header', {
        'header_init-models': initModels,
        header_light: isHeaderLight || (!isDesktop && offerTypeLayout),
        header_dark: type === 'dark',
        header_transparent:
          isHeaderTransparent && !(!isDesktop && offerTypeLayout),
        header_open: isMenuOpen,
        header_search: screenType === 'search',
        header_scroll: !offerTypeLayout && scrollPosition > 0,
        header_hide:
          !offerTypeLayout &&
          hideHeaderOnScroll &&
          scrollPosition > (headerRef?.current?.offsetHeight || 0),
        header_unfixed: !!offerTypeLayout,
        'header_color-black': eventTypeLayout,
      })}
      onMouseLeave={onModelsLeave}
    >
      <div className="container">
        <div className="row">
          <div className="col-12">
            <div className="header__container">
              {!offerTypeLayout && screenType === 'models' && !isDesktop && (
                <span className="back-button" onClick={handleBackButton}>
                  {Arrow}
                </span>
              )}

              {!isMenuOpen &&
                (!offerTypeLayout ? (
                  <Link
                    href={`/${lang}`}
                    className="header__logo"
                    onMouseEnter={onModelsLeave}
                  >
                    {CheryLogo}
                  </Link>
                ) : (
                  <div className="header__logo">{CheryLogo}</div>
                ))}
              <div className="header__content">
                {!offerTypeLayout && (
                  <div className="header__content-container">
                    {(isMenuOpen || isDesktop) && (
                      <>
                        <nav className="header__menu-nav">
                          <ul>
                            {data?.navigation?.map(
                              ({ title, url, hideInMobile, hideInDesktop }) => {
                                if (!isDesktop && hideInMobile) return;
                                if (isDesktop && hideInDesktop) return;
                                return (
                                  <li key={title}>
                                    {url.endsWith('models') ? (
                                      <span
                                        style={{ cursor: 'pointer' }}
                                        className={cn(
                                          'typo-body-1-m header__menu-item',
                                          {
                                            'header__menu-item_active':
                                              screenType === 'models' &&
                                              isDesktop,
                                          },
                                        )}
                                        onClick={handleModelsClick}
                                        onMouseEnter={onModelsEnter}
                                      >
                                        {title}
                                      </span>
                                    ) : (
                                      <Link
                                        href={`/${lang}/${url}`}
                                        className="typo-body-1-m header__menu-item"
                                        onMouseEnter={onModelsLeave}
                                      >
                                        {title}
                                      </Link>
                                    )}
                                  </li>
                                );
                              },
                            )}
                          </ul>
                        </nav>

                        <div className="header__menu-actions">
                          {data?.cta?.navigation?.length && (
                            <MobileCtaButtons
                              className="header__menu-cta"
                              navigation={data?.cta?.navigation}
                              lang={lang}
                            />
                          )}
                          <div className="header__menu-settings">
                            {/* <Link href="/account" className="typo-body-1">
                            My account
                          </Link> */}
                            <MobileLanguage
                              className="header__menu-lang"
                              defaultLanguage={lang}
                              languages={locales}
                            />
                          </div>
                        </div>
                      </>
                    )}
                    {(screenType === 'models' || isDesktop) && (
                      <ModelsList
                        className={cn('header__models-list', {
                          'header__models-list_none': !initModels,
                          'header__models-list_active':
                            isDesktop && screenType === 'models',
                        })}
                        models={models}
                        lang={lang}
                        translates={translates}
                      />
                    )}
                  </div>
                )}

                {!offerTypeLayout && !isDesktop && screenType === 'search' && (
                  <div className="header__search-overlay"></div>
                )}
              </div>

              {!offerTypeLayout && screenType === 'models' && !isDesktop && (
                <span className="screen-type typo-body-1-m">
                  {translates?.models || 'Models'}
                </span>
              )}

              <div className="header__actions">
                {!offerTypeLayout && isDesktop && (
                  <>
                    <div className="header__find-dealer">
                      <FindDealer text={translates?.findDealer} />
                    </div>
                    <div className="header__locations">
                      <Link
                        href={`/${lang}/locations`}
                        className="typo-body-1-m header__menu-item"
                        onMouseEnter={onModelsLeave}
                      >
                        {translates?.outNetwork || 'Our network'}
                      </Link>
                    </div>
                  </>
                )}

                {!isMenuOpen && (
                  <div className="header__lang" onMouseEnter={onModelsLeave}>
                    <Language defaultLanguage={lang} languages={locales} />
                  </div>
                )}

                {!offerTypeLayout && (
                  <>
                    <span
                      className="header__loupe"
                      onClick={() => {
                        handleSearchOpen(true);
                      }}
                      onMouseEnter={onModelsLeave}
                    >
                      {Loupe}
                    </span>
                    <div className="header__burger">
                      <Burger
                        open={isMenuOpen}
                        onBurgerClick={handleMenuClick}
                      />
                    </div>
                  </>
                )}
              </div>
            </div>
            {!offerTypeLayout && screenType === 'search' && (
              <Search
                translates={translates}
                className="header__search"
                onSearchClose={() => handleSearchOpen(false)}
              />
            )}
          </div>
        </div>
      </div>
    </header>
  );
};

export default Header;
