'use client';

import './index.scss';

import { useForm } from 'react-hook-form';
import Input from '@/shared/ui/Input/index';
import Checkbox from '@/shared/ui/Checkbox/index';
import Select from '@/shared/ui/Select/index';
import Button from '@/shared/ui/Button/index';
import Datepicker from '@/shared/ui/Datepicker/index';
import RichText from '@/shared/ui/RichText';
import cn from 'classnames';
import Switcher from '@/shared/ui/Switcher';
import OtpInput from '@/shared/ui/OtpInput';
import PhoneInput from '@/shared/ui/PhoneInput';
import { formatDate } from '@/app/lib/utils';
import UploadInput from '@/shared/ui/UploadInput';
import { useState } from 'react';
import dynamic from 'next/dynamic';

const DynamicModal = dynamic(() => import('../Modal'));

interface FormInput {
  id?: number;
  __component:
    | 'form.input'
    | 'form.select'
    | 'form.datepicker'
    | 'form.checkbox'
    | 'form.switcher'
    | 'form.otp-input'
    | 'form.phone-input'
    | 'form.file-input';
  required: boolean;
  placeholder: string;
  label: string;
  name: string;
  pattern?: { regex: string; message: string };
  type?: 'input' | 'textarea';
  options?: { [key: string]: string } | { [key: string]: string }[];
  errors?: { [key: string]: string };
  extensions?: any;
  hint?: string;
  selectedValue?: string;
  fileSize?: string;
  maxLength?: number;
}

export interface IForm {
  data: {
    title?: string;
    description?: string;
    uid?: string;
    inputs: FormInput[];
    submit: { title: string; endpoint: string; caption?: string };
    success: {
      title: string;
      description: string;
      button: { title: string };
    };
    errors?: { [key: string]: string };
    backButton?: string;
  };
  onSuccessButtonClick?: () => void;
  step?: boolean;
  onBackButton?: () => void;
  onNextButton?: (data: any) => void;
  values?: { [key: string]: string };
  privacyModal?: {
    linkTrigger: string;
    text: string;
  };
}

const Form = ({
  data,
  onSuccessButtonClick = () => {},
  step,
  onNextButton,
  onBackButton,
  values = {},
  privacyModal,
}: IForm) => {
  const { title, description } = data || {};
  const [privacyModalOpen, setPrivacyModalOpen] = useState(false);
  const {
    register,
    formState: { errors: formErrors, isValid },
    handleSubmit,
    setValue,
  } = useForm({ mode: 'onBlur' });

  const onSubmit = async (data: any) => {
    try {
      if (step) {
        onNextButton?.(data);
      } else {
        await fetch(data.submit.endpoint, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(data),
        });
      }
    } catch (e) {
      console.log(e);
    }
  };

  const innerFieldContent = (input: FormInput) => {
    const {
      __component,
      errors = {},
      id,
      label,
      name,
      placeholder,
      required,
      type,
      pattern,
      hint,
      extensions,
      maxLength,
    } = input;

    let options: any = {};

    if (maxLength) {
      options['maxLength'] = {
        value: +maxLength,
        message:
          errors?.maxLength || data.errors?.maxLength || 'Limit exceeded',
      };
    }

    if (required) {
      options['required'] = {
        value: required,
        message:
          errors?.required || data.errors?.required || 'Field is required',
      };
    }

    if (pattern?.regex) {
      options['pattern'] = {
        value: new RegExp(pattern.regex),
        message: pattern.message,
      };
    }

    const value = values[name] || '';

    options.value = value;

    const error = formErrors[name]?.message as string;

    const registeredParams = register(name, options);

    switch (__component) {
      case 'form.select':
        let activeOption;

        const options = Object.entries(input.options || []).map(
          ([val, text]) => {
            const option = {
              value: val,
              text,
            };
            if (val === value) {
              //@ts-ignore
              activeOption = option;
            }
            return option;
          },
        );

        return (
          <Select
            key={name}
            name={name}
            placeholder={placeholder}
            label={label}
            value={activeOption}
            options={options}
            hint={hint}
            error={error}
            onChange={(value: string) => {
              setValue(name, value, { shouldValidate: true });
            }}
          />
        );
      case 'form.datepicker': {
        return (
          <Datepicker
            key={name}
            placeholder={placeholder}
            error={error}
            label={label}
            //@ts-ignore
            value={value}
            hint={hint}
            onChange={(value: Date) => {
              setValue(name, formatDate(value), { shouldValidate: true });
            }}
          />
        );
      }
      case 'form.checkbox':
        return (
          <Checkbox
            key={name}
            label={label}
            error={error}
            value={value}
            useMakeLinksExternal
            onLabelClick={(e: any) => {
              if (
                privacyModal?.linkTrigger &&
                e.target?.href?.includes(privacyModal.linkTrigger)
              ) {
                e.preventDefault();
                setPrivacyModalOpen(true);
              }
            }}
            {...registeredParams}
          />
        );

      case 'form.switcher':
        return (
          <Switcher
            key={name}
            error={error}
            label={label}
            hint={hint}
            selectedValue={value || input.selectedValue || ''}
            options={input.options as []}
            onChange={(value: string) => {
              setValue(name, value, { shouldValidate: true });
            }}
          />
        );
      case 'form.otp-input':
        return (
          <OtpInput
            key={name}
            label={label}
            hint={hint}
            error={error}
            value={value} //@ts-ignore
            length={input.codeLength}
            onChange={(value) => {
              setValue(name, value, { shouldValidate: !!value });
            }}
            onBlur={(value) => {
              setValue(name, value, { shouldValidate: !!required });
            }}
          />
        );
      case 'form.phone-input':
        //@ts-ignore
        const mask = input.options[0].mask;
        return (
          <PhoneInput
            key={name}
            label={label}
            hint={hint}
            error={error}
            placeholder={mask}
            mask={mask}
            value={value}
            onChange={(value) => {
              setValue(name, value, { shouldValidate: !!value });
            }}
            onBlur={(value) => {
              setValue(name, value, { shouldValidate: !!required });
            }}
          />
        );
      case 'form.file-input':
        const size = input.fileSize;

        return (
          <UploadInput
            key={name}
            name={name}
            placeholder={placeholder}
            label={label}
            hint={hint}
            value={value ? (value as unknown as File) : null}
            error={error}
            extensions={(extensions || []).map(({ text }: any) => text)}
            fileSize={size ? +size : 2}
            onChange={(file: File | null) => {
              setValue(name, file, { shouldValidate: !!required });
            }}
          />
        );
      case 'form.input':
      default:
        return (
          <Input
            key={name}
            placeholder={placeholder}
            type={type}
            label={label}
            hint={hint}
            value={value}
            error={error}
            maxLength={maxLength}
            {...registeredParams}
          />
        );
    }
  };

  return data.inputs.length ? (
    <form className="form" onSubmit={handleSubmit(onSubmit)} noValidate>
      {title && (
        <div className="form__title">
          <RichText text={title} />
        </div>
      )}
      {description && (
        <div className="form__description">
          <RichText text={description} />
        </div>
      )}
      <div className="form__inputs">
        <div className="form__inputs-container">
          {data.inputs.map(innerFieldContent)}
        </div>
      </div>
      <div
        className={cn('form__buttons', {
          form__buttons_two: step && data.backButton,
        })}
      >
        {data.submit?.caption && (
          <RichText
            className="form__submit-caption"
            text={data.submit.caption}
          />
        )}
        {step && data.backButton && (
          <Button
            className={cn('form__back')}
            border={true}
            appearance={'transparent'}
            {...{ onClick: onBackButton }}
          >
            <span className="typo-link-big">{data.backButton}</span>
          </Button>
        )}
        <Button
          className={cn('form__submit', { disabled: !isValid })}
          {...{ type: 'submit', disabled: !isValid }}
        >
          <span className="typo-link-big">{data.submit?.title}</span>
        </Button>
      </div>
      {privacyModal?.text && (
        <DynamicModal
          open={privacyModalOpen}
          onClose={() => setPrivacyModalOpen(false)}
          className={'form__modal'}
        >
          <RichText text={privacyModal?.text} />
        </DynamicModal>
      )}
    </form>
  ) : (
    <div className="form-success">
      <div className="form-success__container">
        <div className="form-success__icon">
          <svg
            xmlns="http://www.w3.org/2000/svg"
            viewBox="0 0 41 41"
            fill="none"
          >
            <rect
              x="1.297"
              y="1.5"
              width="38"
              height="38"
              rx="19"
              stroke="currentColor"
              strokeWidth="2"
            />
            <path
              d="M12.5228 19.413L18.3393 25.2295L28.2211 15.2852"
              stroke="currentColor"
              strokeWidth="2"
            />
          </svg>
        </div>
        <div className="form-success__text">
          <span className="form-success__title typo-head-2">
            {data.success?.title}
          </span>
          <p className="form-success__desciption typo-body-1">
            {data.success?.description}
          </p>
        </div>

        {data.success?.button?.title && (
          <Button
            className="form-success__button"
            {...{
              onClick: onSuccessButtonClick,
            }}
          >
            <span className="typo-link-big">{data.success?.button?.title}</span>
          </Button>
        )}
      </div>
    </div>
  );
};

export default Form;
