import { ChangeEvent, useEffect, useRef, useState } from 'react';
import cn from 'classnames';
import { Cross, UploadIcon } from '@/app/assets/icons';
import './index.scss';

interface IUploadInput {
  label?: string;
  placeholder?: string;
  error?: string;
  value?: File | null;
  name: string;
  hint?: string;
  extensions: string[];
  fileSize: number;
  onChange?: (file: File | null) => void;
}

const UploadInput = ({
  label = '',
  placeholder = '',
  error = '',
  value,
  name,
  hint,
  extensions,
  fileSize,
  onChange = () => {},
}: IUploadInput) => {
  const ref = useRef(null);
  const [drag, setDrag] = useState(false);
  const [file, setFile] = useState<File | null>(null);

  useEffect(() => {
    if (value) {
      setFile(value);
    }
  }, []);

  const ALLOWED_EXTENSIONS = extensions;
  const ALLOWED_SIZE_MB = fileSize;
  const B_TO_MB = 1024 * 1024;

  const hasExtension = (name: string) => {
    return new RegExp(
      '(' + ALLOWED_EXTENSIONS.join('|').replace(/\./g, '\\.') + ')$',
      'i',
    ).test(name);
  };

  const removePinnedFile = () => {
    //@ts-ignore
    ref.current.value = null;
    setFile(null);

    onChange(null);
  };

  const handleChange = (event: ChangeEvent<HTMLInputElement>) => {
    if (!event?.target?.files?.[0]) return;
    const file = event.target.files[0];
    const { name, size } = file;

    if (file) {
      try {
        if (ALLOWED_EXTENSIONS !== null && !hasExtension(name)) {
          throw new Error();
        }

        if (ALLOWED_SIZE_MB !== null && size > ALLOWED_SIZE_MB * B_TO_MB) {
          throw new Error();
        }

        setFile(file);
        onChange(file);
      } catch (e) {
        removePinnedFile();
      }

      setDrag(false);
    }
  };

  const handleDragEnter = () => {
    setDrag(true);
  };

  const handleDragLeave = () => {
    setDrag(false);
  };

  const handleIconClick = () => {
    if (file) {
      removePinnedFile();
    }
  };

  return (
    <div
      className={cn('upload-input', {
        'upload-input_has-file': file,
        'upload-input_drag': drag,
      })}
    >
      <div className="upload-input__container">
        {label && (
          <label className="upload-input__label typo-body-3">{label}</label>
        )}
        <div className="upload-input__content">
          <input
            ref={ref}
            type="file"
            accept={ALLOWED_EXTENSIONS.join(',.')}
            name={name}
            onChange={handleChange}
            onDragOver={handleDragEnter}
            onDragLeave={handleDragLeave}
          />
          <div className="upload-input__text">
            <span className="upload-input__placeholder typo-body-1-m">
              {file?.name || placeholder}
            </span>
            {hint && !file?.name && (
              <span className="upload-input__hint typo-body-3">{hint}</span>
            )}
          </div>
          <div className="upload-input__icon" onClick={handleIconClick}>
            {!file ? UploadIcon : Cross}
          </div>
        </div>

        {error && (
          <span className="upload-input__error typo-body-3">{error}</span>
        )}
      </div>
    </div>
  );
};

export default UploadInput;
