import { ChangeEvent, forwardRef, useEffect, useRef, useState } from 'react';
import './index.scss';
import cn from 'classnames';
import useClickOutside from '@/app/lib/use-outside-click';

interface ISwitcher {
  label?: string;
  error?: string;
  hint?: string;
  options: { value: string; name: string }[];
  selectedValue: string;
  onChange?: (value: string) => void;
}

const Switcher = forwardRef<HTMLDivElement, ISwitcher>(
  (
    {
      label = '',
      error = '',
      hint,
      selectedValue,
      options = [],
      onChange = () => {},
    },
    ref,
  ) => {
    const getInitVal = () => {
      const option = options.findIndex(({ value }) => value === selectedValue);
      return Boolean(Number(option === -1 ? 0 : option));
    };

    const [isChecked, setIsChecked] = useState(getInitVal());

    const handleChange = (e: ChangeEvent<HTMLInputElement>) => {
      setIsChecked(e.target.checked);
      const checked = Number(e.target.checked);
      onChange(options[checked].value);
    };

    useEffect(() => {
      onChange(options[Number(isChecked)].value);
    }, []);

    return (
      <div className="switcher">
        <div className="switcher__container">
          {label && (
            <label className="select__label typo-body-3">{label}</label>
          )}

          <div className="switcher__content">
            <label className={cn('switch', { switch_checked: isChecked })}>
              <div className="switcher__options ">
                {options.map(({ value, name }) => (
                  <span className="typo-body-2" key={value}>
                    {name}
                  </span>
                ))}
              </div>
              <input
                type="checkbox"
                onChange={handleChange}
                checked={isChecked}
              />
              <span className="slider"></span>
            </label>
          </div>
          {hint && <span className="switcher__hint typo-body-3">{hint}</span>}
          {error && (
            <span className="switcher__error typo-body-3">{error}</span>
          )}
        </div>
      </div>
    );
  },
);

Switcher.displayName = 'Switcher';
export default Switcher;
