import {
  ChangeEvent,
  forwardRef,
  use,
  useEffect,
  useRef,
  useState,
} from 'react';
import './index.scss';
import cn from 'classnames';
import useClickOutside from '@/app/lib/use-outside-click';

interface ISelect {
  label?: string;
  error?: string;
  hideDefaultPlaceholder?: boolean;
  defaultValue?: string;
  value?: { value: string; text: string };
  placeholder?: string;
  name: string;
  hint?: string;
  options: { value: string; text: string }[];
  onChange?: (value: string) => void;
}

const Select = forwardRef<HTMLSelectElement, ISelect>(
  (
    {
      label = '',
      error = '',
      placeholder = '',
      hint,
      options = [],
      value,
      hideDefaultPlaceholder,
      onChange = () => {},
    },
    ref,
  ) => {
    const DEFAULT_PLACEHOLDER = { text: placeholder, value: '' };
    const modifiedOptions = [DEFAULT_PLACEHOLDER, ...options];
    if (hideDefaultPlaceholder) {
      modifiedOptions.shift();
    }
    const [searchValue, setSearchValue] = useState('');
    const [selected, setSelected] = useState(value || DEFAULT_PLACEHOLDER);
    const [showItems, setShowItems] = useState(false);
    const containerRef = useRef(null);
    const [searchChanged, setSearchChanged] = useState(false);

    useEffect(() => {
      if (value?.value) {
        setSearchValue(value.text);
      }
    }, []);

    //@ts-ignore
    useClickOutside(containerRef, () => {
      if (showItems) {
        setShowItems(false);
        !selected?.value && setSearchValue('');
        setSearchChanged(false);
      }
    });

    return (
      <div className={cn('select', { select_open: showItems })}>
        <div className="select__container">
          {label && (
            <label className="select__label typo-body-3">{label}</label>
          )}
          <div className="select__content" ref={containerRef}>
            <div className="select__search">
              <input
                className="select__option typo-body-1-m"
                placeholder={placeholder}
                value={searchValue}
                onChange={(e) => {
                  const value = e.target.value;
                  setSearchValue(value);
                  setSearchChanged(true);
                }}
                onFocus={() => {
                  setShowItems(true);
                }}
                autoComplete="dsbld"
              />

              <svg
                xmlns="http://www.w3.org/2000/svg"
                width="10"
                height="5"
                viewBox="0 0 10 5"
                fill="none"
                className="select__indicator"
              >
                <path
                  d="M5 4.93872L0.5 0.438721H9.5L5 4.93872Z"
                  fill="currentColor"
                />
              </svg>
            </div>

            <div
              className={cn('select__items', { select__items_show: showItems })}
            >
              <div className="select__items-container">
                {modifiedOptions
                  .filter(({ text, value }) => {
                    return searchValue && searchChanged
                      ? value
                        ? text.toLowerCase().includes(searchValue.toLowerCase())
                        : false
                      : true;
                  })
                  .map(({ text, value }) => (
                    <span
                      key={value}
                      className={cn({ active: value === selected.value })}
                      onClick={() => {
                        setSearchValue(!value ? value : text);
                        setSelected({ text, value });
                        setShowItems(false);
                        setSearchChanged(false);
                        onChange(value);
                      }}
                    >
                      {text}
                    </span>
                  ))}
              </div>
            </div>
          </div>
          {hint && <span className="select__hint typo-body-3">{hint}</span>}
          {error && <span className="select__error typo-body-3">{error}</span>}
        </div>
      </div>
    );
  },
);
Select.displayName = 'Select';
export default Select;
