'use client';

import { CUSTOMER_ID } from '@/app/constants/common';
import { EVENTS } from '@/app/lib/constants';
import { sendTrackingEvent } from '@/app/lib/send-tracking-event';
import { usePathname } from 'next/navigation';
import React, { useEffect } from 'react';
import { useExitIntent } from 'use-exit-intent';

export function ScarabEventTracker() {
  const pathname = usePathname();
  let prevPathname: string | null = '';

  const { registerHandler: registerExitIntentHandler } = useExitIntent({
    desktop: {
      triggerOnIdle: true,
      useBeforeUnload: false,
      triggerOnMouseLeave: true,
      delayInSecondsToTrigger: 10,
    },
    mobile: {
      triggerOnIdle: true,
      delayInSecondsToTrigger: 10,
    },
  });

  const handlePathChange = (pathname: string | null): void => {
    if (!pathname) {
      return;
    }

    const modelPattern = /\/([a-z]{2})\/models\/(.*)/;
    const resultsPattern = /\/([a-z]{2})\/results/;

    const generatedId = localStorage.getItem(CUSTOMER_ID);

    if (generatedId) {
      sendTrackingEvent([EVENTS.SET_CUSTOMER_ID, generatedId]);
    }

    if (modelPattern.test(pathname)) {
      const [_, _locale, modelSlug] = pathname.match(modelPattern) || [];
      sendTrackingEvent([EVENTS.VIEW, modelSlug]);
    }

    if (resultsPattern.test(pathname)) {
      const url = new URL(window.location.href);
      const searchQuery: string = url.searchParams.get('q')!;
      sendTrackingEvent([EVENTS.SEARCH_TERM, searchQuery]);
    }

    sendTrackingEvent([EVENTS.GO]);
  };

  useEffect(() => {
    if (pathname !== prevPathname) {
      handlePathChange(pathname);
    }
    prevPathname = pathname;
  }, [pathname]);

  const discoverHandler = (e: MouseEvent) => {
    //@ts-ignore
    const url = e.target?.href;

    if (!url?.includes('models')) return;
    sendTrackingEvent([
      EVENTS.TAG,
      'cta_click',
      {
        button: 'Discover',
      },
    ]);
    sendTrackingEvent([EVENTS.GO]);
  };

  registerExitIntentHandler({
    id: 'track-exit-intent',
    handler: () => {
      sendTrackingEvent([EVENTS.TAG, 'exit_intent']);

      let curtime;

      if (window.currentPage) {
        curtime = new Date().getTime();

        sendTrackingEvent([
          EVENTS.TAG,
          'time_on_page',
          {
            ...window.currentPage,
            time_spent: (curtime - window.currentPage.time_spent) / 1000,
          },
        ]);

        window.currentPage = null;
      }

      if (window.currentSection) {
        curtime = curtime || new Date().getTime();

        sendTrackingEvent([
          EVENTS.TAG,
          'scroll_depth',
          {
            ...window.currentSection,
            time_spent: (curtime - window.currentSection.time_spent) / 1000,
          },
        ]);

        window.currentSection = null;
      }
      sendTrackingEvent([EVENTS.GO]);
    },
  });

  useEffect(() => {
    window.addEventListener('click', discoverHandler);

    return () => {
      window.removeEventListener('click', discoverHandler);
    };
  }, []);

  return <></>;
}
