const axios = require("axios");
const NodeCache = require("node-cache");

const tokenCache = new NodeCache();
const TOKEN_CACHE_KEY = "authToken";
let tokenPromise = null;

const getOAuth2Token = async ({ tokenUrl, clientId, clientSecret }) => {
  try {
    const params = new URLSearchParams();
    params.append("grant_type", "client_credentials");

    const credentials = Buffer.from(`${clientId}:${clientSecret}`).toString(
      "base64"
    );

    const res = await axios.post(tokenUrl, params.toString(), {
      headers: {
        Authorization: `Basic ${credentials}`,
        "Content-Type": "application/x-www-form-urlencoded",
      },
    });

    if (res.status !== 200) {
      throw new Error(`Failed to get token. Status: ${res.status}`);
    }

    const { access_token, expires_in } = res.data;

    console.log("New token expires in:", expires_in);

    tokenCache.set(TOKEN_CACHE_KEY, access_token, expires_in || 0);
    return access_token;
  } catch (error) {
    console.error("Error fetching OAuth2 token:", error.message);
    throw new Error("Could not retrieve OAuth2 token");
  }
};

const getToken = async ({ tokenUrl, clientId, clientSecret }) => {
  let token = tokenCache.get(TOKEN_CACHE_KEY);

  if (token) {
    return token;
  }

  if (!tokenPromise) {
    console.log("No valid or expired token found, fetching new one...");
    tokenPromise = getOAuth2Token({ tokenUrl, clientId, clientSecret });
  }

  try {
    token = await tokenPromise;
    return token;
  } catch (error) {
    console.error("Failed to fetch a new token:", error.message);
    throw error;
  } finally {
    tokenPromise = null;
  }
};

module.exports = async (requestBody) => {
  const url = process.env.CDP_API_EVENT_ENDPOINT_URL;

  if (!url) {
    throw new Error("CDP API event endpoint URL isn`t defined");
  }

  const tokenUrl = process.env.CDP_API_OAUTH_TOKEN_URL;
  if (!tokenUrl) {
    throw new Error("tokenUrl isn`t defined");
  }

  const clientId = process.env.CDP_API_CLIENT_ID;
  if (!clientId) {
    throw new Error("clientId isn`t defined");
  }

  const clientSecret = process.env.CDP_API_CLIENT_SECRET;
  if (!clientSecret) {
    throw new Error("clientSecret isn`t defined");
  }

  const token = await getToken({ tokenUrl, clientId, clientSecret });
  if (!token) {
    throw new Error("Failed to get token");
  }

  try {
    await axios.post(url, requestBody, {
      headers: {
        "Content-Type": "application/json",
        Authorization: `Bearer ${token}`,
      },
    });
  } catch (error) {
    if (error.response && [401, 403].includes(error.response.status)) {
      console.log("Cached token is invalid, resetting cache and retrying...");
      tokenCache.del(TOKEN_CACHE_KEY);
      return;
    }

    throw error;
  }
};
