import { pipeline } from "stream";
import fs from "fs";
import path from "path";
import fse from "fs-extra";
import * as utils from "@strapi/utils";
const { PayloadTooLargeError } = utils.errors;
const { kbytesToBytes, bytesToHumanReadable } = utils.file;
const UPLOADS_FOLDER_NAME = "uploads";
const index = {
  init({ sizeLimit: providerOptionsSizeLimit } = {}) {
    if (providerOptionsSizeLimit) {
      process.emitWarning(
        '[deprecated] In future versions, "sizeLimit" argument will be ignored from upload.config.providerOptions. Move it to upload.config'
      );
    }
    const uploadPath = path.resolve(strapi.dirs.static.public, UPLOADS_FOLDER_NAME);
    if (!fse.pathExistsSync(uploadPath)) {
      throw new Error(
        `The upload folder (${uploadPath}) doesn't exist or is not accessible. Please make sure it exists.`
      );
    }
    return {
      checkFileSize(file, options) {
        const { sizeLimit } = options ?? {};
        if (providerOptionsSizeLimit) {
          if (kbytesToBytes(file.size) > providerOptionsSizeLimit)
            throw new PayloadTooLargeError(
              `${file.name} exceeds size limit of ${bytesToHumanReadable(
                providerOptionsSizeLimit
              )}.`
            );
        } else if (sizeLimit) {
          if (kbytesToBytes(file.size) > sizeLimit)
            throw new PayloadTooLargeError(
              `${file.name} exceeds size limit of ${bytesToHumanReadable(sizeLimit)}.`
            );
        }
      },
      uploadStream(file) {
        if (!file.stream) {
          return Promise.reject(new Error("Missing file stream"));
        }
        const { stream } = file;
        return new Promise((resolve, reject) => {
          pipeline(
            stream,
            fs.createWriteStream(path.join(uploadPath, `${file.hash}${file.ext}`)),
            (err) => {
              if (err) {
                return reject(err);
              }
              file.url = `/${UPLOADS_FOLDER_NAME}/${file.hash}${file.ext}`;
              resolve();
            }
          );
        });
      },
      upload(file) {
        if (!file.buffer) {
          return Promise.reject(new Error("Missing file buffer"));
        }
        const { buffer } = file;
        return new Promise((resolve, reject) => {
          fs.writeFile(path.join(uploadPath, `${file.hash}${file.ext}`), buffer, (err) => {
            if (err) {
              return reject(err);
            }
            file.url = `/${UPLOADS_FOLDER_NAME}/${file.hash}${file.ext}`;
            resolve();
          });
        });
      },
      delete(file) {
        return new Promise((resolve, reject) => {
          const filePath = path.join(uploadPath, `${file.hash}${file.ext}`);
          if (!fs.existsSync(filePath)) {
            resolve("File doesn't exist");
            return;
          }
          fs.unlink(filePath, (err) => {
            if (err) {
              return reject(err);
            }
            resolve();
          });
        });
      }
    };
  }
};
export {
  index as default
};
//# sourceMappingURL=index.mjs.map
