import * as React from 'react';
import { useCallbackRef } from '@strapi/ui-primitives';

function useControllableState({ prop, defaultProp, onChange = () => { }, }) {
    const [uncontrolledProp, setUncontrolledProp] = useUncontrolledState({ defaultProp, onChange });
    const isControlled = prop !== undefined;
    const propValue = prop instanceof Function ? prop(uncontrolledProp) : prop;
    const value = isControlled ? propValue : uncontrolledProp;
    const handleChange = useCallbackRef(onChange);
    const setValue = React.useCallback((nextValue) => {
        if (isControlled) {
            const setter = nextValue;
            const value = typeof nextValue === 'function' ? setter(propValue) : nextValue;
            if (value !== propValue) {
                handleChange(value);
                setUncontrolledProp(nextValue);
            }
        }
        else {
            setUncontrolledProp(nextValue);
        }
    }, [isControlled, propValue, setUncontrolledProp, handleChange]);
    return [value, setValue];
}
function useUncontrolledState({ defaultProp, onChange }) {
    const uncontrolledState = React.useState(defaultProp);
    const [value] = uncontrolledState;
    const prevValueRef = React.useRef(value);
    const handleChange = useCallbackRef(onChange);
    React.useEffect(() => {
        if (prevValueRef.current !== value) {
            handleChange(value);
            prevValueRef.current = value;
        }
    }, [value, prevValueRef, handleChange]);
    return uncontrolledState;
}

export { useControllableState };
