'use strict';

Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const jsxRuntime = require('react/jsx-runtime');
const React = require('react');
const uiPrimitives = require('@strapi/ui-primitives');
const styled = require('styled-components');
const components = require('./components.js');
const TabsContext = require('./TabsContext.js');
const keyboardKeys = require('../helpers/keyboardKeys.js');
const Typography = require('../Typography/Typography.js');

const _interopDefault = e => e && e.__esModule ? e : { default: e };

const styled__default = /*#__PURE__*/_interopDefault(styled);

const useTabsFocus = (selectedTabIndex, onTabChange) => {
    const tabsRef = React.useRef(null);
    const mountedRef = React.useRef(false);
    const handleTabChange = uiPrimitives.useCallbackRef(onTabChange);
    React.useEffect(() => {
        if (!tabsRef.current)
            return;
        // We don't' want to send the focus to the tab when it mounts
        // It could break the navigating flow of the users if the focus was supposed to be
        // on another element
        if (mountedRef.current) {
            const nextFocusEl = tabsRef.current.querySelector('[tabindex="0"]');
            if (nextFocusEl) {
                nextFocusEl.focus();
                handleTabChange(selectedTabIndex);
            }
        }
        if (!mountedRef.current) {
            mountedRef.current = true;
        }
    }, [selectedTabIndex, handleTabChange]);
    return tabsRef;
};
const TabButton = styled__default.default.button `
  &[aria-disabled='true'] {
    cursor: not-allowed;
  }
`;
const Tabs = ({ children, ...props }) => {
    const { id, selectedTabIndex, selectTabIndex, label, variant, onTabChange } = TabsContext.useTabs();
    const tabsRef = useTabsFocus(selectedTabIndex, onTabChange);
    const childrenArray = React.Children.toArray(children).map((node, index) => React.cloneElement(node, {
        id: `${id}-${index}`,
        index,
        selectedTabIndex,
        onTabClick: () => selectTabIndex(index),
        variant,
    }));
    const handleKeyDown = (e) => {
        const hasAllChildrenDisabled = childrenArray.every((node) => node.props.disabled);
        if (hasAllChildrenDisabled) {
            return;
        }
        switch (e.key) {
            case keyboardKeys.KeyboardKeys.RIGHT: {
                const nextWantedIndex = selectedTabIndex + 1;
                const findNextIndex = (ref) => {
                    const isDisabled = childrenArray[ref].props.disabled;
                    if (!isDisabled) {
                        return ref;
                    }
                    if (ref === childrenArray.length - 1) {
                        return findNextIndex(0);
                    }
                    return findNextIndex(ref + 1);
                };
                const nextIndex = findNextIndex(nextWantedIndex >= childrenArray.length ? 0 : nextWantedIndex);
                selectTabIndex(nextIndex);
                break;
            }
            case keyboardKeys.KeyboardKeys.LEFT: {
                const nextWantedIndex = selectedTabIndex - 1;
                const findNextIndex = (ref) => {
                    const isDisabled = childrenArray[ref].props.disabled;
                    if (!isDisabled) {
                        return ref;
                    }
                    if (ref === 0) {
                        return findNextIndex(childrenArray.length - 1);
                    }
                    return findNextIndex(ref - 1);
                };
                const nextIndex = findNextIndex(nextWantedIndex < 0 ? childrenArray.length - 1 : nextWantedIndex);
                selectTabIndex(nextIndex);
                break;
            }
            case keyboardKeys.KeyboardKeys.HOME: {
                const nextIndex = childrenArray.findIndex((node) => !node.props.disabled);
                selectTabIndex(nextIndex);
                break;
            }
            case keyboardKeys.KeyboardKeys.END: {
                const arrayOfChildrenProps = childrenArray.map((node, index) => ({ isDisabled: node.props.disabled, index }));
                const firstNonDisabledChildren = arrayOfChildrenProps.reverse().find(({ isDisabled }) => !isDisabled);
                if (firstNonDisabledChildren) {
                    selectTabIndex(firstNonDisabledChildren.index);
                }
                break;
            }
        }
    };
    if (variant === 'simple') {
        return (
        // TODO: This needs to be reviewed how to handle correctly since it's supposed to have focus.
        // eslint-disable-next-line jsx-a11y/interactive-supports-focus
        jsxRuntime.jsx("div", { ref: tabsRef, role: "tablist", "aria-label": label, onKeyDown: handleKeyDown, ...props, children: childrenArray }));
    }
    return (jsxRuntime.jsx(components.DefaultTabsRow, { ref: tabsRef, role: "tablist", alignItems: "flex-end", "aria-label": label, onKeyDown: handleKeyDown, ...props, children: childrenArray }));
};
const Tab = ({ disabled = false, id, children, variant, hasError = false, index, selectedTabIndex, onTabClick, ...props }) => {
    const tabId = `${id}-tab`;
    const tabPanelId = `${id}-tabpanel`;
    const selected = index === selectedTabIndex;
    const handleClick = () => {
        if (disabled) {
            return;
        }
        if (onTabClick) {
            onTabClick();
        }
    };
    if (variant === 'simple') {
        let textColor;
        if (hasError) {
            textColor = 'danger600';
        }
        else if (selected) {
            textColor = 'primary600';
        }
        else if (disabled) {
            textColor = 'neutral600';
        }
        else {
            textColor = 'neutral600';
        }
        return (jsxRuntime.jsx(TabButton, { id: tabId, role: "tab", "aria-controls": selected ? tabPanelId : undefined, tabIndex: selected ? 0 : -1, "aria-selected": selected, type: "button", onClick: handleClick, "aria-disabled": disabled, ...props, children: jsxRuntime.jsx(components.SimpleTabBox, { padding: 4, selected: selected, hasError: hasError, children: jsxRuntime.jsx(Typography.Typography, { variant: "sigma", textColor: textColor, children: children }) }) }));
    }
    if (hasError) {
        console.warn('The "hasError" prop is only available for the "simple" variant.');
    }
    const showRightBorder = selectedTabIndex && selectedTabIndex - 1 === index;
    return (jsxRuntime.jsx(components.DefaultTabButton, { id: tabId, role: "tab", type: "button", "aria-controls": selected ? tabPanelId : undefined, tabIndex: selected ? 0 : -1, "aria-selected": selected, onClick: handleClick, "aria-disabled": disabled, showRightBorder: Boolean(showRightBorder), ...props, children: jsxRuntime.jsx(components.DefaultTabBox, { padding: selected ? 4 : 3, background: selected ? 'neutral0' : 'neutral100', selected: selected, children: jsxRuntime.jsx(Typography.Typography, { fontWeight: "bold", textColor: selected ? 'primary700' : 'neutral600', children: children }) }) }));
};

exports.Tab = Tab;
exports.Tabs = Tabs;
