import { jsx, jsxs } from 'react/jsx-runtime';
import * as React from 'react';
import { useFloating, offset, shift, flip, autoUpdate } from '@floating-ui/react-dom';
import { FocusScope } from '@radix-ui/react-focus-scope';
import { useCallbackRef, composeEventHandlers } from '@strapi/ui-primitives';
import { hideOthers } from 'aria-hidden';
import { RemoveScroll } from 'react-remove-scroll';
import styled from 'styled-components';
import { DismissibleLayer } from '../DismissibleLayer/index.mjs';
import { stripReactIdOfColon } from '../helpers/strings.mjs';
import { useComposedRefs } from '../hooks/useComposeRefs.mjs';
import { useId } from '../hooks/useId.mjs';
import { useIntersection } from '../hooks/useIntersection.mjs';
import { Box } from '../Box/Box.mjs';
import { Portal } from '../Portal/Portal.mjs';

const POPOVER_PLACEMENTS = [
    'top',
    'top-start',
    'top-end',
    'right',
    'right-start',
    'right-end',
    'bottom',
    'bottom-start',
    'bottom-end',
    'left',
    'left-start',
    'left-end',
];
const PopoverWrapper = styled(Box) `
  box-shadow: ${({ theme }) => theme.shadows.filterShadow};
  z-index: ${({ theme }) => theme.zIndices[0]};
  border: 1px solid ${({ theme }) => theme.colors.neutral150};
`;
const Content = React.forwardRef(({ source, children, spacing = 0, fullWidth = false, placement = 'bottom-start', centered = false, onEscapeKeyDown, onPointerDownOutside, onDismiss, onFocusOutside, ...props }, forwardedRef) => {
    const [content, setContent] = React.useState(null);
    const [width, setWidth] = React.useState(undefined);
    const isRightClickOutsideRef = React.useRef(false);
    const { x, y, refs, strategy } = useFloating({
        strategy: 'fixed',
        placement: centered ? 'bottom' : placement,
        middleware: [
            offset({
                mainAxis: spacing,
            }),
            shift(),
            flip(),
        ],
        elements: {
            reference: source.current,
        },
        whileElementsMounted: autoUpdate,
    });
    React.useLayoutEffect(() => {
        if (fullWidth) {
            setWidth(source.current.offsetWidth);
        }
    }, [fullWidth, source]);
    // aria-hide everything except the content (better supported equivalent to setting aria-modal)
    React.useEffect(() => {
        if (content)
            return hideOthers(content);
    }, [content]);
    const handleDismiss = useCallbackRef(onDismiss);
    React.useEffect(() => {
        const close = () => {
            handleDismiss();
        };
        window.addEventListener('blur', close);
        window.addEventListener('resize', close);
        return () => {
            window.removeEventListener('blur', close);
            window.removeEventListener('resize', close);
        };
    }, [handleDismiss]);
    const composedRefs = useComposedRefs(forwardedRef, (node) => setContent(node), refs.setFloating);
    return (jsx(RemoveScroll, { allowPinchZoom: true, children: jsx(FocusScope, { asChild: true, loop: true, 
            // we make sure we're not trapping once it's been closed
            // (closed !== unmounted when animating out)
            trapped: true, onUnmountAutoFocus: (event) => {
                event.preventDefault();
                if (!isRightClickOutsideRef.current)
                    source.current?.focus({ preventScroll: true });
            }, children: jsx(DismissibleLayer, { asChild: true, onEscapeKeyDown: onEscapeKeyDown, onPointerDownOutside: composeEventHandlers(onPointerDownOutside, (event) => {
                    const originalEvent = event.detail.originalEvent;
                    const ctrlLeftClick = originalEvent.button === 0 && originalEvent.ctrlKey === true;
                    const isRightClick = originalEvent.button === 2 || ctrlLeftClick;
                    isRightClickOutsideRef.current = isRightClick;
                }, { checkForDefaultPrevented: false }), 
                // When focus is trapped, a `focusout` event may still happen.
                // We make sure we don't trigger our `onDismiss` in such case.
                onFocusOutside: composeEventHandlers(onFocusOutside, (event) => event.preventDefault(), {
                    checkForDefaultPrevented: false,
                }), onDismiss: onDismiss, children: jsx(PopoverWrapper, { ref: composedRefs, style: {
                        left: x,
                        top: y,
                        position: strategy,
                        width: width || undefined,
                    }, hasRadius: true, background: "neutral0", padding: 1, ...props, children: children }) }) }) }));
});
const Scrolling = ({ children, intersectionId, onReachEnd, ...props }) => {
    const popoverRef = React.useRef(null);
    const generatedIntersectionId = useId();
    useIntersection(popoverRef, onReachEnd ?? (() => { }), {
        selectorToWatch: `#${stripReactIdOfColon(generatedIntersectionId)}`,
        skipWhen: !intersectionId || !onReachEnd,
    });
    return (jsxs(PopoverScrollable, { ref: popoverRef, ...props, children: [children, intersectionId && onReachEnd && (jsx(Box, { id: stripReactIdOfColon(generatedIntersectionId), width: "100%", height: "1px" }))] }));
};
const PopoverScrollable = styled(Box) `
  // 16 is base base size, 3 is the factor to get closer to 40px and 5 is the number of elements visible in the list
  max-height: ${3 * 5}rem;
  overflow-y: auto;
  overflow-x: hidden;

  &::-webkit-scrollbar {
    -webkit-appearance: none;
    width: 4px;
  }

  &::-webkit-scrollbar-track {
    background: ${({ theme }) => theme.colors.neutral0};
  }

  &::-webkit-scrollbar-thumb {
    background: ${({ theme }) => theme.colors.neutral150};
    border-radius: ${({ theme }) => theme.borderRadius};
    margin-right: 10px;
  }
`;
const Popover = ({ children, source, spacing, fullWidth, placement, centered, onEscapeKeyDown, onPointerDownOutside, onDismiss, ...restProps }) => {
    return (jsx(Portal, { children: jsx(Content, { source: source, spacing: spacing, fullWidth: fullWidth, placement: placement, centered: centered, onEscapeKeyDown: onEscapeKeyDown, onPointerDownOutside: onPointerDownOutside, onDismiss: onDismiss, children: jsx(Scrolling, { ...restProps, children: children }) }) }));
};

export { Content, POPOVER_PLACEMENTS, Popover, Scrolling };
