import { jsx } from 'react/jsx-runtime';
import { KeyboardKeys } from '../helpers/keyboardKeys.mjs';
import { Box } from '../Box/Box.mjs';

const KeyboardNavigable = ({ tagName, attributeName = '', ...props }) => {
    const isValidFocusedElement = () => {
        const focused = document.activeElement;
        if (!focused) {
            return false;
        }
        if (tagName) {
            return focused.tagName.toLowerCase() === tagName;
        }
        return focused.hasAttribute(attributeName);
    };
    const queryElement = (parentEl) => {
        if (tagName) {
            return parentEl.querySelectorAll(tagName);
        }
        return parentEl.querySelectorAll(`[${attributeName}]`);
    };
    const handleKeyDown = (e) => {
        switch (e.key) {
            case KeyboardKeys.RIGHT:
            case KeyboardKeys.DOWN: {
                if (isValidFocusedElement()) {
                    e.preventDefault();
                    const focused = document.activeElement;
                    const allElements = [...queryElement(e.currentTarget)];
                    const focusedIndex = allElements.findIndex((node) => node === focused);
                    const nextIndex = focusedIndex + 1 < allElements.length ? focusedIndex + 1 : 0;
                    allElements[nextIndex].focus();
                }
                break;
            }
            case KeyboardKeys.LEFT:
            case KeyboardKeys.UP: {
                if (isValidFocusedElement()) {
                    e.preventDefault();
                    const focused = document.activeElement;
                    const allElements = [...queryElement(e.currentTarget)];
                    const focusedIndex = allElements.findIndex((node) => node === focused);
                    const nextIndex = focusedIndex - 1 > -1 ? focusedIndex - 1 : allElements.length - 1;
                    allElements[nextIndex].focus();
                }
                break;
            }
            case KeyboardKeys.HOME: {
                if (isValidFocusedElement()) {
                    e.preventDefault();
                    const allElements = queryElement(e.currentTarget);
                    const focusElement = allElements.item(0);
                    focusElement.focus();
                }
                break;
            }
            case KeyboardKeys.END: {
                if (isValidFocusedElement()) {
                    e.preventDefault();
                    const allElements = queryElement(e.currentTarget);
                    const focusElement = allElements.item(allElements.length - 1);
                    focusElement.focus();
                }
                break;
            }
        }
    };
    return jsx(Box, { onKeyDown: handleKeyDown, ...props });
};

export { KeyboardNavigable };
