'use strict';

Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const jsxRuntime = require('react/jsx-runtime');
const React = require('react');
const getFocusableNodes = require('../helpers/getFocusableNodes.js');
const keyboardKeys = require('../helpers/keyboardKeys.js');

const FocusTrap = ({ onEscape, restoreFocus = true, skipAutoFocus = false, ...props }) => {
    const trappedRef = React.useRef(null);
    /**
     * Restore the focus to the previously focused element (often, it's the CTA that opened the trap)
     */
    React.useEffect(() => {
        let currentFocus = null;
        if (restoreFocus) {
            currentFocus = document.activeElement;
        }
        return () => {
            if (currentFocus) {
                currentFocus.focus();
            }
        };
    }, [restoreFocus]);
    /**
     * Sends the focus to the first element of the focus trap tree
     */
    React.useEffect(() => {
        if (!trappedRef.current || skipAutoFocus)
            return;
        const focusableChildren = getFocusableNodes.getFocusableNodes(trappedRef.current);
        if (focusableChildren.length > 0) {
            const firstElement = focusableChildren[0];
            firstElement.focus();
        }
        else {
            console.warn('[FocusTrap]: it seems there are no focusable elements in the focus trap tree. Make sure there s at least one.');
        }
    }, [skipAutoFocus]);
    const handleKeyDown = (e) => {
        if (e.key === keyboardKeys.KeyboardKeys.ESCAPE && onEscape) {
            onEscape();
            return;
        }
        if (e.key !== keyboardKeys.KeyboardKeys.TAB)
            return;
        const focusableChildren = getFocusableNodes.getFocusableNodes(trappedRef.current);
        if (focusableChildren.length > 0) {
            const firstElement = focusableChildren[0];
            const lastElement = focusableChildren[focusableChildren.length - 1];
            // e.shiftKey allows to verify reverse tab
            if (e.shiftKey) {
                if (firstElement === document.activeElement) {
                    e.preventDefault();
                    lastElement.focus();
                }
            }
            else if (lastElement === document.activeElement) {
                e.preventDefault();
                firstElement.focus();
            }
        }
    };
    // eslint-disable-next-line jsx-a11y/no-static-element-interactions
    return jsxRuntime.jsx("div", { ref: trappedRef, onKeyDown: handleKeyDown, ...props });
};

exports.FocusTrap = FocusTrap;
