import { jsxs, jsx } from 'react/jsx-runtime';
import * as React from 'react';
import { Cross, CarretDown } from '@strapi/icons';
import { Combobox as Combobox$1 } from '@strapi/ui-primitives';
import styled from 'styled-components';
import { stripReactIdOfColon } from '../helpers/strings.mjs';
import { useComposedRefs } from '../hooks/useComposeRefs.mjs';
import { useControllableState } from '../hooks/useControllableState.mjs';
import { useId } from '../hooks/useId.mjs';
import { useIntersection } from '../hooks/useIntersection.mjs';
import { Loader } from '../Loader/Loader.mjs';
import { getThemeSize, inputFocusStyle } from '../themes/utils.mjs';
import { Flex } from '../Flex/Flex.mjs';
import { Box } from '../Box/Box.mjs';
import { Typography } from '../Typography/Typography.mjs';
import { Field } from '../Field/Field.mjs';
import { FieldLabel } from '../Field/FieldLabel.mjs';
import { FieldHint } from '../Field/FieldHint.mjs';
import { FieldError } from '../Field/FieldError.mjs';

const ComboboxInput = React.forwardRef(({ allowCustomValue, autocomplete, children, className, clearLabel = 'clear', creatable = false, createMessage = (value) => `Create "${value}"`, defaultFilterValue, defaultTextValue, defaultOpen = false, open, onOpenChange, disabled = false, error, filterValue, hasMoreItems = false, id, isPrintableCharacter, loading = false, loadingMessage = 'Loading content...', noOptionsMessage = () => 'No results found', onChange, onClear, onCreateOption, onFilterValueChange, onInputChange, onTextValueChange, onLoadMore, placeholder = 'Select or enter a value', required = false, size = 'M', startIcon, textValue, value, ...restProps }, forwardedRef) => {
    const [internalIsOpen, setInternalIsOpen] = useControllableState({
        prop: open,
        defaultProp: defaultOpen,
        onChange: onOpenChange,
    });
    const [internalTextValue, setInternalTextValue] = useControllableState({
        prop: textValue,
        defaultProp: allowCustomValue && !defaultTextValue ? value : defaultTextValue,
        onChange: onTextValueChange,
    });
    const [internalFilterValue, setInternalFilterValue] = useControllableState({
        prop: filterValue,
        defaultProp: defaultFilterValue,
        onChange: onFilterValueChange,
    });
    /**
     * Used for the intersection observer
     */
    const viewportRef = React.useRef(null);
    const triggerRef = React.useRef(null);
    const composedTriggerRefs = useComposedRefs(triggerRef, forwardedRef);
    const clearRef = React.useRef(null);
    const handleClearClick = (e) => {
        if (onClear && !disabled) {
            setInternalTextValue('');
            setInternalFilterValue('');
            onClear(e);
            triggerRef.current.focus();
        }
    };
    const handleOpenChange = (open) => {
        setInternalIsOpen(open);
    };
    const handleTextValueChange = (textValue) => {
        setInternalTextValue(textValue);
    };
    const handleFilterValueChange = (filterValue) => {
        setInternalFilterValue(filterValue);
    };
    const handleInputChange = (e) => {
        if (onInputChange) {
            onInputChange(e);
        }
    };
    const handleChange = (value) => {
        if (onChange) {
            onChange(value);
        }
    };
    const handleReachEnd = (entry) => {
        if (onLoadMore && hasMoreItems && !loading) {
            onLoadMore(entry);
        }
    };
    const handleCreateItemClick = () => {
        if (onCreateOption && internalTextValue) {
            onCreateOption(internalTextValue);
        }
    };
    const generatedId = useId(id);
    const generatedIntersectionId = useId();
    const intersectionId = `intersection-${stripReactIdOfColon(generatedIntersectionId)}`;
    useIntersection(viewportRef, handleReachEnd, {
        selectorToWatch: `#${intersectionId}`,
        /**
         * We need to know when the select is open because only then will viewportRef
         * not be null. Because it uses a portal that (sensibly) is not mounted 24/7.
         */
        skipWhen: !internalIsOpen,
    });
    const hintId = `${generatedId}-hint`;
    const errorId = `${generatedId}-error`;
    return (jsxs(Combobox$1.Root, { autocomplete: autocomplete || (creatable ? 'list' : 'both'), onOpenChange: handleOpenChange, open: internalIsOpen, onTextValueChange: handleTextValueChange, textValue: internalTextValue, allowCustomValue: creatable || allowCustomValue, disabled: disabled, required: required, value: value, onValueChange: handleChange, filterValue: internalFilterValue, onFilterValueChange: handleFilterValueChange, isPrintableCharacter: isPrintableCharacter, children: [jsxs(Trigger, { "$hasError": Boolean(error), "$size": size, className: className, children: [jsxs(Flex, { flex: "1", as: "span", gap: 3, children: [startIcon ? (jsx(Box, { as: "span", "aria-hidden": true, children: startIcon })) : null, jsx(TextInput, { placeholder: placeholder, id: id, "aria-invalid": Boolean(error), "aria-describedby": `${hintId} ${errorId}`, onChange: handleInputChange, ref: composedTriggerRefs, ...restProps })] }), jsxs(Flex, { as: "span", gap: 3, children: [internalTextValue && onClear ? (jsx(IconBox, { as: "button", hasRadius: true, background: "transparent", type: "button", onClick: handleClearClick, "aria-disabled": disabled, "aria-label": clearLabel, title: clearLabel, ref: clearRef, children: jsx(Cross, {}) })) : null, jsx(DownIcon, { children: jsx(CarretDown, {}) })] })] }), jsx(Combobox$1.Portal, { children: jsx(Content, { sideOffset: 4, children: jsxs(Viewport, { ref: viewportRef, children: [children, creatable ? (jsx(Combobox$1.CreateItem, { onPointerUp: handleCreateItemClick, onClick: handleCreateItemClick, asChild: true, children: jsx(OptionBox, { children: jsx(Typography, { children: createMessage(internalTextValue ?? '') }) }) })) : null, !creatable && !loading ? (jsx(Combobox$1.NoValueFound, { asChild: true, children: jsx(OptionBox, { "$hasHover": false, children: jsx(Typography, { children: noOptionsMessage(internalTextValue ?? '') }) }) })) : null, loading ? (jsx(Flex, { justifyContent: "center", alignItems: "center", paddingTop: 2, paddingBottom: 2, children: jsx(Loader, { small: true, children: loadingMessage }) })) : null, jsx(Box, { id: intersectionId, width: "100%", height: "1px" })] }) }) })] }));
});
const Combobox = React.forwardRef(({ error, hint, id, label, labelAction, required = false, ...restProps }, forwardedRef) => {
    const generatedId = useId(id);
    return (jsx(Field, { hint: hint, error: error, id: generatedId, required: required, children: jsxs(Flex, { direction: "column", alignItems: "stretch", gap: 1, children: [label ? jsx(FieldLabel, { action: labelAction, children: label }) : null, jsx(ComboboxInput, { ref: forwardedRef, id: generatedId, error: error, required: required, ...restProps }), jsx(FieldHint, {}), jsx(FieldError, {})] }) }));
});
const CreatableCombobox = (props) => jsx(Combobox, { ...props, creatable: true });
const IconBox = styled(Box) `
  border: none;

  svg {
    height: ${11 / 16}rem;
    width: ${11 / 16}rem;
  }

  svg path {
    fill: ${({ theme }) => theme.colors.neutral600};
  }
`;
const Trigger = styled(Combobox$1.Trigger) `
  position: relative;
  border: 1px solid ${({ theme, $hasError }) => ($hasError ? theme.colors.danger600 : theme.colors.neutral200)};
  padding-right: ${({ theme }) => theme.spaces[3]};
  padding-left: ${({ theme }) => theme.spaces[3]};
  border-radius: ${({ theme }) => theme.borderRadius};
  background: ${({ theme }) => theme.colors.neutral0};
  overflow: hidden;
  display: flex;
  align-items: center;
  justify-content: space-between;
  gap: ${({ theme }) => theme.spaces[4]};
  min-height: ${({ theme, $size }) => getThemeSize('input')({ theme, size: $size })};

  &[data-disabled] {
    color: ${({ theme }) => theme.colors.neutral600};
    background: ${({ theme }) => theme.colors.neutral150};
    cursor: not-allowed;
  }

  /* Required to ensure the below inputFocusStyles are adhered too */
  &:focus-visible {
    outline: none;
  }

  ${({ theme, $hasError }) => inputFocusStyle()({ theme, hasError: $hasError })};
`;
const TextInput = styled(Combobox$1.TextInput) `
  width: 100%;
  font-size: ${14 / 16}rem;
  color: ${({ theme }) => theme.colors.neutral800};
  padding: 0;
  border: none;
  background-color: transparent;

  &:focus-visible {
    outline: none;
  }

  &[aria-disabled='true'] {
    cursor: inherit;
  }
`;
const DownIcon = styled(Combobox$1.Icon) `
  & > svg {
    width: ${6 / 16}rem;

    & > path {
      fill: ${({ theme }) => theme.colors.neutral600};
    }
  }

  &[aria-disabled='true'] {
    cursor: inherit;
  }
`;
const Content = styled(Combobox$1.Content) `
  background: ${({ theme }) => theme.colors.neutral0};
  box-shadow: ${({ theme }) => theme.shadows.filterShadow};
  border: 1px solid ${({ theme }) => theme.colors.neutral150};
  border-radius: ${({ theme }) => theme.borderRadius};
  width: var(--radix-combobox-trigger-width);
  /* This is from the design-system figma file. */
  max-height: 15rem;
  z-index: ${({ theme }) => theme.zIndices[1]};
`;
const Viewport = styled(Combobox$1.Viewport) `
  padding: ${({ theme }) => theme.spaces[1]};
`;
const Option = React.forwardRef(({ children, value, disabled, textValue, ...props }, ref) => {
    return (jsx(Combobox$1.ComboboxItem, { asChild: true, value: value, disabled: disabled, textValue: textValue, children: jsx(OptionBox, { ref: ref, ...props, children: jsx(Combobox$1.ItemText, { asChild: true, children: jsx(Typography, { children: children }) }) }) }));
});
const OptionBox = styled.div `
  width: 100%;
  border: none;
  text-align: left;
  outline-offset: -3px;
  padding: ${({ theme }) => theme.spaces[2]} ${({ theme }) => theme.spaces[4]};
  background-color: ${({ theme }) => theme.colors.neutral0};
  border-radius: ${({ theme }) => theme.borderRadius};
  user-select: none;

  &[data-state='checked'] {
    background-color: ${({ theme }) => theme.colors.primary100};

    ${Typography} {
      color: ${({ theme }) => theme.colors.primary600};
      font-weight: bold;
    }
  }

  &:hover,
  &[data-highlighted] {
    outline: none;
    background-color: ${({ theme, $hasHover = true }) => ($hasHover ? theme.colors.primary100 : theme.colors.neutral0)};
  }

  &[data-highlighted] {
    ${Typography} {
      color: ${({ theme }) => theme.colors.primary600};
      font-weight: bold;
    }
  }
`;

export { Combobox, ComboboxInput, CreatableCombobox, Option, OptionBox };
